<?php
require_once __DIR__ . '/../config/config.php';
require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../helpers/functions.php';
require_once __DIR__ . '/../models/MovimentoFinanceiro.php';
require_once __DIR__ . '/../models/CentroCusto.php';
require_once __DIR__ . '/../models/Igreja.php';

class FinanceiroController
{
    private $movimentoModel;
    private $centroCustoModel;
    private $igrejaModel;

    public function __construct()
    {
        verificarLogin();
        $this->movimentoModel = new MovimentoFinanceiro();
        $this->centroCustoModel = new CentroCusto();
        $this->igrejaModel = new Igreja();
    }

    public function index()
    {
        $filtros = [];
        
        if (!empty($_GET['data_inicio'])) {
            $filtros['data_inicio'] = $_GET['data_inicio'];
        } else {
            $filtros['data_inicio'] = date('Y-m-01');
        }
        
        if (!empty($_GET['data_fim'])) {
            $filtros['data_fim'] = $_GET['data_fim'];
        } else {
            $filtros['data_fim'] = date('Y-m-t');
        }
        
        if (!empty($_GET['tipo'])) {
            $filtros['tipo'] = $_GET['tipo'];
        }
        
        if (!empty($_GET['id_igreja'])) {
            $filtros['id_igreja'] = intval($_GET['id_igreja']);
        }
        
        if (!empty($_GET['id_centro_custo'])) {
            $filtros['id_centro_custo'] = intval($_GET['id_centro_custo']);
        }

        $movimentos = $this->movimentoModel->listarTodos($filtros);
        $saldo = $this->movimentoModel->calcularSaldo($filtros);
        $igrejas = $this->igrejaModel->listarTodos(true);
        $centrosCusto = $this->centroCustoModel->listarTodos();
        
        include __DIR__ . '/../views/layouts/header.php';
        include __DIR__ . '/../views/layouts/sidebar.php';
        include __DIR__ . '/../views/financeiro/index.php';
        include __DIR__ . '/../views/layouts/footer.php';
    }

    public function criar()
    {
        $igrejas = $this->igrejaModel->listarTodos(true);
        $centrosCustoEntrada = $this->centroCustoModel->listarTodos('ENTRADA');
        $centrosCustoSaida = $this->centroCustoModel->listarTodos('SAIDA');

        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $dados = [
                'data_movimento' => $_POST['data_movimento'] ?? date('Y-m-d'),
                'tipo' => $_POST['tipo'] ?? 'ENTRADA',
                'valor' => floatval(str_replace(',', '.', str_replace('.', '', $_POST['valor'] ?? '0'))),
                'descricao' => sanitizar($_POST['descricao'] ?? ''),
                'id_centro_custo' => intval($_POST['id_centro_custo'] ?? 0),
                'id_igreja' => !empty($_POST['id_igreja']) ? intval($_POST['id_igreja']) : null,
                'forma_pagamento' => sanitizar($_POST['forma_pagamento'] ?? ''),
                'observacoes' => sanitizar($_POST['observacoes'] ?? ''),
                'criado_por' => $_SESSION['usuario_id']
            ];

            if (empty($dados['data_movimento']) || empty($dados['valor']) || empty($dados['id_centro_custo'])) {
                setMensagemErro('Data, valor e centro de custo são obrigatórios.');
            } elseif ($dados['valor'] <= 0) {
                setMensagemErro('O valor deve ser maior que zero.');
            } else {
                if ($this->movimentoModel->criar($dados)) {
                    registrarLog('CREATE', "Movimento financeiro criado: {$dados['tipo']} - R$ " . number_format($dados['valor'], 2, ',', '.'));
                    setMensagemSucesso('Movimento financeiro registrado com sucesso!');
                    redirecionar('index.php?c=Financeiro&a=index');
                    return;
                } else {
                    setMensagemErro('Erro ao registrar movimento financeiro.');
                }
            }
        }

        include __DIR__ . '/../views/layouts/header.php';
        include __DIR__ . '/../views/layouts/sidebar.php';
        include __DIR__ . '/../views/financeiro/form.php';
        include __DIR__ . '/../views/layouts/footer.php';
    }

    public function editar()
    {
        $id = $_GET['id'] ?? 0;
        $movimento = $this->movimentoModel->buscarPorId($id);
        $igrejas = $this->igrejaModel->listarTodos(true);
        $centrosCustoEntrada = $this->centroCustoModel->listarTodos('ENTRADA');
        $centrosCustoSaida = $this->centroCustoModel->listarTodos('SAIDA');

        if (!$movimento) {
            setMensagemErro('Movimento financeiro não encontrado.');
            redirecionar('index.php?c=Financeiro&a=index');
            return;
        }

        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $dados = [
                'data_movimento' => $_POST['data_movimento'] ?? date('Y-m-d'),
                'tipo' => $_POST['tipo'] ?? 'ENTRADA',
                'valor' => floatval(str_replace(',', '.', str_replace('.', '', $_POST['valor'] ?? '0'))),
                'descricao' => sanitizar($_POST['descricao'] ?? ''),
                'id_centro_custo' => intval($_POST['id_centro_custo'] ?? 0),
                'id_igreja' => !empty($_POST['id_igreja']) ? intval($_POST['id_igreja']) : null,
                'forma_pagamento' => sanitizar($_POST['forma_pagamento'] ?? ''),
                'observacoes' => sanitizar($_POST['observacoes'] ?? '')
            ];

            if (empty($dados['data_movimento']) || empty($dados['valor']) || empty($dados['id_centro_custo'])) {
                setMensagemErro('Data, valor e centro de custo são obrigatórios.');
            } elseif ($dados['valor'] <= 0) {
                setMensagemErro('O valor deve ser maior que zero.');
            } else {
                if ($this->movimentoModel->atualizar($id, $dados)) {
                    registrarLog('UPDATE', "Movimento financeiro atualizado: {$dados['tipo']} - R$ " . number_format($dados['valor'], 2, ',', '.') . " (ID: {$id})");
                    setMensagemSucesso('Movimento financeiro atualizado com sucesso!');
                    redirecionar('index.php?c=Financeiro&a=index');
                    return;
                } else {
                    setMensagemErro('Erro ao atualizar movimento financeiro.');
                }
            }
        }

        include __DIR__ . '/../views/layouts/header.php';
        include __DIR__ . '/../views/layouts/sidebar.php';
        include __DIR__ . '/../views/financeiro/form.php';
        include __DIR__ . '/../views/layouts/footer.php';
    }

    public function excluir()
    {
        $id = $_GET['id'] ?? 0;
        $movimento = $this->movimentoModel->buscarPorId($id);

        if (!$movimento) {
            setMensagemErro('Movimento financeiro não encontrado.');
        } else {
            if ($this->movimentoModel->excluir($id)) {
                registrarLog('DELETE', "Movimento financeiro excluído (ID: {$id})");
                setMensagemSucesso('Movimento financeiro excluído com sucesso!');
            } else {
                setMensagemErro('Erro ao excluir movimento financeiro.');
            }
        }

        redirecionar('index.php?c=Financeiro&a=index');
    }

    public function relatorio()
    {
        $filtros = [];
        
        if (!empty($_GET['data_inicio'])) {
            $filtros['data_inicio'] = $_GET['data_inicio'];
        } else {
            $filtros['data_inicio'] = date('Y-m-01');
        }
        
        if (!empty($_GET['data_fim'])) {
            $filtros['data_fim'] = $_GET['data_fim'];
        } else {
            $filtros['data_fim'] = date('Y-m-t');
        }
        
        if (!empty($_GET['id_igreja'])) {
            $filtros['id_igreja'] = intval($_GET['id_igreja']);
        }

        $saldo = $this->movimentoModel->calcularSaldo($filtros);
        $resumoPorCentroCusto = $this->movimentoModel->resumoPorCentroCusto($filtros);
        $igrejas = $this->igrejaModel->listarTodos(true);
        
        include __DIR__ . '/../views/layouts/header.php';
        include __DIR__ . '/../views/layouts/sidebar.php';
        include __DIR__ . '/../views/financeiro/relatorio.php';
        include __DIR__ . '/../views/layouts/footer.php';
    }
}

