<?php
require_once __DIR__ . '/../config/config.php';
require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../helpers/functions.php';
require_once __DIR__ . '/../models/Obreiro.php';
require_once __DIR__ . '/../models/Igreja.php';

class ObreiroController
{
    private $obreiroModel;
    private $igrejaModel;

    public function __construct()
    {
        verificarLogin();
        $this->obreiroModel = new Obreiro();
        $this->igrejaModel = new Igreja();
    }

    public function index()
    {
        $filtros = [];
        
        if (!empty($_GET['nome'])) {
            $filtros['nome'] = sanitizar($_GET['nome']);
        }
        
        if (!empty($_GET['tipo'])) {
            $filtros['tipo'] = sanitizar($_GET['tipo']);
        }
        
        if (!empty($_GET['id_igreja'])) {
            $filtros['id_igreja'] = intval($_GET['id_igreja']);
        }
        
        if (isset($_GET['ativo'])) {
            $filtros['ativo'] = intval($_GET['ativo']);
        } else {
            $filtros['ativo'] = 1;
        }

        $obreiros = $this->obreiroModel->listarTodos($filtros);
        $igrejas = $this->igrejaModel->listarTodos(true);
        $tipos = $this->obreiroModel->listarTipos();
        
        include __DIR__ . '/../views/layouts/header.php';
        include __DIR__ . '/../views/layouts/sidebar.php';
        include __DIR__ . '/../views/obreiros/index.php';
        include __DIR__ . '/../views/layouts/footer.php';
    }

    public function criar()
    {
        $igrejas = $this->igrejaModel->listarTodos(true);
        $tipos = $this->obreiroModel->listarTipos();

        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $dados = [
                'nome' => sanitizar($_POST['nome'] ?? ''),
                'cpf' => sanitizar($_POST['cpf'] ?? ''),
                'tipo' => sanitizar($_POST['tipo'] ?? ''),
                'telefone' => sanitizar($_POST['telefone'] ?? ''),
                'email' => sanitizar($_POST['email'] ?? ''),
                'endereco' => sanitizar($_POST['endereco'] ?? ''),
                'cidade' => sanitizar($_POST['cidade'] ?? ''),
                'estado' => sanitizar($_POST['estado'] ?? ''),
                'cep' => sanitizar($_POST['cep'] ?? ''),
                'id_igreja_atual' => !empty($_POST['id_igreja_atual']) ? intval($_POST['id_igreja_atual']) : null,
                'data_consagracao' => !empty($_POST['data_consagracao']) ? $_POST['data_consagracao'] : null,
                'observacoes' => sanitizar($_POST['observacoes'] ?? ''),
                'ativo' => isset($_POST['ativo']) ? 1 : 0
            ];

            if (empty($dados['nome']) || empty($dados['tipo'])) {
                setMensagemErro('Nome e tipo são obrigatórios.');
            } else {
                if ($this->obreiroModel->criar($dados)) {
                    registrarLog('CREATE', "Obreiro criado: {$dados['nome']}");
                    setMensagemSucesso('Obreiro cadastrado com sucesso!');
                    redirecionar('index.php?c=Obreiro&a=index');
                    return;
                } else {
                    setMensagemErro('Erro ao cadastrar obreiro.');
                }
            }
        }

        include __DIR__ . '/../views/layouts/header.php';
        include __DIR__ . '/../views/layouts/sidebar.php';
        include __DIR__ . '/../views/obreiros/form.php';
        include __DIR__ . '/../views/layouts/footer.php';
    }

    public function editar()
    {
        $id = $_GET['id'] ?? 0;
        $obreiro = $this->obreiroModel->buscarPorId($id);
        $igrejas = $this->igrejaModel->listarTodos(true);
        $tipos = $this->obreiroModel->listarTipos();

        if (!$obreiro) {
            setMensagemErro('Obreiro não encontrado.');
            redirecionar('index.php?c=Obreiro&a=index');
            return;
        }

        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $dados = [
                'nome' => sanitizar($_POST['nome'] ?? ''),
                'cpf' => sanitizar($_POST['cpf'] ?? ''),
                'tipo' => sanitizar($_POST['tipo'] ?? ''),
                'telefone' => sanitizar($_POST['telefone'] ?? ''),
                'email' => sanitizar($_POST['email'] ?? ''),
                'endereco' => sanitizar($_POST['endereco'] ?? ''),
                'cidade' => sanitizar($_POST['cidade'] ?? ''),
                'estado' => sanitizar($_POST['estado'] ?? ''),
                'cep' => sanitizar($_POST['cep'] ?? ''),
                'id_igreja_atual' => !empty($_POST['id_igreja_atual']) ? intval($_POST['id_igreja_atual']) : null,
                'data_consagracao' => !empty($_POST['data_consagracao']) ? $_POST['data_consagracao'] : null,
                'observacoes' => sanitizar($_POST['observacoes'] ?? ''),
                'ativo' => isset($_POST['ativo']) ? 1 : 0
            ];

            if (empty($dados['nome']) || empty($dados['tipo'])) {
                setMensagemErro('Nome e tipo são obrigatórios.');
            } else {
                if ($this->obreiroModel->atualizar($id, $dados)) {
                    registrarLog('UPDATE', "Obreiro atualizado: {$dados['nome']} (ID: {$id})");
                    setMensagemSucesso('Obreiro atualizado com sucesso!');
                    redirecionar('index.php?c=Obreiro&a=index');
                    return;
                } else {
                    setMensagemErro('Erro ao atualizar obreiro.');
                }
            }
        }

        include __DIR__ . '/../views/layouts/header.php';
        include __DIR__ . '/../views/layouts/sidebar.php';
        include __DIR__ . '/../views/obreiros/form.php';
        include __DIR__ . '/../views/layouts/footer.php';
    }

    public function visualizar()
    {
        $id = $_GET['id'] ?? 0;
        $obreiro = $this->obreiroModel->buscarPorId($id);

        if (!$obreiro) {
            setMensagemErro('Obreiro não encontrado.');
            redirecionar('index.php?c=Obreiro&a=index');
            return;
        }

        // Buscar histórico de transferências
        require_once __DIR__ . '/../models/Transferencia.php';
        $transferenciaModel = new Transferencia();
        $historico = $transferenciaModel->historicoObreiro($id);

        include __DIR__ . '/../views/layouts/header.php';
        include __DIR__ . '/../views/layouts/sidebar.php';
        include __DIR__ . '/../views/obreiros/show.php';
        include __DIR__ . '/../views/layouts/footer.php';
    }

    public function excluir()
    {
        $id = $_GET['id'] ?? 0;
        $obreiro = $this->obreiroModel->buscarPorId($id);

        if (!$obreiro) {
            setMensagemErro('Obreiro não encontrado.');
        } else {
            if ($this->obreiroModel->excluir($id)) {
                registrarLog('DELETE', "Obreiro excluído: {$obreiro['nome']} (ID: {$id})");
                setMensagemSucesso('Obreiro excluído com sucesso!');
            } else {
                setMensagemErro('Erro ao excluir obreiro.');
            }
        }

        redirecionar('index.php?c=Obreiro&a=index');
    }
}

