<?php
require_once __DIR__ . '/../config/config.php';
require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../helpers/functions.php';
require_once __DIR__ . '/../models/Pastor.php';
require_once __DIR__ . '/../models/Igreja.php';

class PastorController
{
    private $pastorModel;
    private $igrejaModel;

    public function __construct()
    {
        verificarLogin();
        $this->pastorModel = new Pastor();
        $this->igrejaModel = new Igreja();
    }

    public function index()
    {
        $filtros = [];
        
        if (!empty($_GET['nome'])) {
            $filtros['nome'] = sanitizar($_GET['nome']);
        }
        
        if (!empty($_GET['id_igreja'])) {
            $filtros['id_igreja'] = intval($_GET['id_igreja']);
        }
        
        if (isset($_GET['ativo'])) {
            $filtros['ativo'] = intval($_GET['ativo']);
        } else {
            $filtros['ativo'] = 1;
        }

        $pastores = $this->pastorModel->listarTodos($filtros);
        $igrejas = $this->igrejaModel->listarTodos(true);
        
        include __DIR__ . '/../views/layouts/header.php';
        include __DIR__ . '/../views/layouts/sidebar.php';
        include __DIR__ . '/../views/pastores/index.php';
        include __DIR__ . '/../views/layouts/footer.php';
    }

    public function criar()
    {
        $igrejas = $this->igrejaModel->listarTodos(true);

        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $dados = [
                'nome' => sanitizar($_POST['nome'] ?? ''),
                'cpf' => sanitizar($_POST['cpf'] ?? ''),
                'rg' => sanitizar($_POST['rg'] ?? ''),
                'data_nascimento' => !empty($_POST['data_nascimento']) ? $_POST['data_nascimento'] : null,
                'estado_civil' => sanitizar($_POST['estado_civil'] ?? ''),
                'telefone' => sanitizar($_POST['telefone'] ?? ''),
                'email' => sanitizar($_POST['email'] ?? ''),
                'endereco' => sanitizar($_POST['endereco'] ?? ''),
                'cidade' => sanitizar($_POST['cidade'] ?? ''),
                'estado' => sanitizar($_POST['estado'] ?? ''),
                'cep' => sanitizar($_POST['cep'] ?? ''),
                'id_igreja_atual' => !empty($_POST['id_igreja_atual']) ? intval($_POST['id_igreja_atual']) : null,
                'data_ordenacao' => !empty($_POST['data_ordenacao']) ? $_POST['data_ordenacao'] : null,
                'observacoes' => sanitizar($_POST['observacoes'] ?? ''),
                'ativo' => isset($_POST['ativo']) ? 1 : 0
            ];

            if (empty($dados['nome']) || empty($dados['cpf'])) {
                setMensagemErro('Nome e CPF são obrigatórios.');
            } elseif (!validarCPF($dados['cpf'])) {
                setMensagemErro('CPF inválido.');
            } else {
                // Verificar se CPF já existe
                $pastorExistente = $this->pastorModel->buscarPorCPF($dados['cpf']);
                if ($pastorExistente) {
                    setMensagemErro('Já existe um pastor cadastrado com este CPF.');
                } else {
                    if ($this->pastorModel->criar($dados)) {
                        registrarLog('CREATE', "Pastor criado: {$dados['nome']}");
                        setMensagemSucesso('Pastor cadastrado com sucesso!');
                        redirecionar('index.php?c=Pastor&a=index');
                        return;
                    } else {
                        setMensagemErro('Erro ao cadastrar pastor.');
                    }
                }
            }
        }

        include __DIR__ . '/../views/layouts/header.php';
        include __DIR__ . '/../views/layouts/sidebar.php';
        include __DIR__ . '/../views/pastores/form.php';
        include __DIR__ . '/../views/layouts/footer.php';
    }

    public function editar()
    {
        $id = $_GET['id'] ?? 0;
        $pastor = $this->pastorModel->buscarPorId($id);
        $igrejas = $this->igrejaModel->listarTodos(true);

        if (!$pastor) {
            setMensagemErro('Pastor não encontrado.');
            redirecionar('index.php?c=Pastor&a=index');
            return;
        }

        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $dados = [
                'nome' => sanitizar($_POST['nome'] ?? ''),
                'cpf' => sanitizar($_POST['cpf'] ?? ''),
                'rg' => sanitizar($_POST['rg'] ?? ''),
                'data_nascimento' => !empty($_POST['data_nascimento']) ? $_POST['data_nascimento'] : null,
                'estado_civil' => sanitizar($_POST['estado_civil'] ?? ''),
                'telefone' => sanitizar($_POST['telefone'] ?? ''),
                'email' => sanitizar($_POST['email'] ?? ''),
                'endereco' => sanitizar($_POST['endereco'] ?? ''),
                'cidade' => sanitizar($_POST['cidade'] ?? ''),
                'estado' => sanitizar($_POST['estado'] ?? ''),
                'cep' => sanitizar($_POST['cep'] ?? ''),
                'id_igreja_atual' => !empty($_POST['id_igreja_atual']) ? intval($_POST['id_igreja_atual']) : null,
                'data_ordenacao' => !empty($_POST['data_ordenacao']) ? $_POST['data_ordenacao'] : null,
                'observacoes' => sanitizar($_POST['observacoes'] ?? ''),
                'ativo' => isset($_POST['ativo']) ? 1 : 0
            ];

            if (empty($dados['nome']) || empty($dados['cpf'])) {
                setMensagemErro('Nome e CPF são obrigatórios.');
            } elseif (!validarCPF($dados['cpf'])) {
                setMensagemErro('CPF inválido.');
            } else {
                // Verificar se CPF já existe em outro pastor
                $pastorExistente = $this->pastorModel->buscarPorCPF($dados['cpf']);
                if ($pastorExistente && $pastorExistente['id'] != $id) {
                    setMensagemErro('Já existe outro pastor cadastrado com este CPF.');
                } else {
                    if ($this->pastorModel->atualizar($id, $dados)) {
                        registrarLog('UPDATE', "Pastor atualizado: {$dados['nome']} (ID: {$id})");
                        setMensagemSucesso('Pastor atualizado com sucesso!');
                        redirecionar('index.php?c=Pastor&a=index');
                        return;
                    } else {
                        setMensagemErro('Erro ao atualizar pastor.');
                    }
                }
            }
        }

        include __DIR__ . '/../views/layouts/header.php';
        include __DIR__ . '/../views/layouts/sidebar.php';
        include __DIR__ . '/../views/pastores/form.php';
        include __DIR__ . '/../views/layouts/footer.php';
    }

    public function visualizar()
    {
        $id = $_GET['id'] ?? 0;
        $pastor = $this->pastorModel->buscarPorId($id);

        if (!$pastor) {
            setMensagemErro('Pastor não encontrado.');
            redirecionar('index.php?c=Pastor&a=index');
            return;
        }

        // Buscar histórico de transferências
        require_once __DIR__ . '/../models/Transferencia.php';
        $transferenciaModel = new Transferencia();
        $historico = $transferenciaModel->historicoPastor($id);

        include __DIR__ . '/../views/layouts/header.php';
        include __DIR__ . '/../views/layouts/sidebar.php';
        include __DIR__ . '/../views/pastores/show.php';
        include __DIR__ . '/../views/layouts/footer.php';
    }

    public function excluir()
    {
        $id = $_GET['id'] ?? 0;
        $pastor = $this->pastorModel->buscarPorId($id);

        if (!$pastor) {
            setMensagemErro('Pastor não encontrado.');
        } else {
            if ($this->pastorModel->excluir($id)) {
                registrarLog('DELETE', "Pastor excluído: {$pastor['nome']} (ID: {$id})");
                setMensagemSucesso('Pastor excluído com sucesso!');
            } else {
                setMensagemErro('Erro ao excluir pastor.');
            }
        }

        redirecionar('index.php?c=Pastor&a=index');
    }
}

