<?php
require_once __DIR__ . '/../config/config.php';
require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../helpers/functions.php';
require_once __DIR__ . '/../models/Usuario.php';

class UsuarioController
{
    private $usuarioModel;

    public function __construct()
    {
        verificarLogin();
        verificarPermissao([1]); // Apenas Admin Geral
        $this->usuarioModel = new Usuario();
    }

    public function index()
    {
        $usuarios = $this->usuarioModel->listarTodos();
        
        include __DIR__ . '/../views/layouts/header.php';
        include __DIR__ . '/../views/layouts/sidebar.php';
        include __DIR__ . '/../views/usuarios/index.php';
        include __DIR__ . '/../views/layouts/footer.php';
    }

    public function criar()
    {
        $perfis = $this->usuarioModel->listarPerfis();

        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $dados = [
                'nome' => sanitizar($_POST['nome'] ?? ''),
                'email' => sanitizar($_POST['email'] ?? ''),
                'senha' => $_POST['senha'] ?? '',
                'id_perfil' => intval($_POST['id_perfil'] ?? 0),
                'ativo' => isset($_POST['ativo']) ? 1 : 0
            ];

            if (empty($dados['nome']) || empty($dados['email']) || empty($dados['senha'])) {
                setMensagemErro('Nome, e-mail e senha são obrigatórios.');
            } elseif (strlen($dados['senha']) < PASSWORD_MIN_LENGTH) {
                setMensagemErro('A senha deve ter no mínimo ' . PASSWORD_MIN_LENGTH . ' caracteres.');
            } elseif (empty($dados['id_perfil'])) {
                setMensagemErro('Selecione um perfil.');
            } else {
                // Verificar se email já existe
                $usuarioExistente = $this->usuarioModel->buscarPorEmail($dados['email']);
                if ($usuarioExistente) {
                    setMensagemErro('Já existe um usuário cadastrado com este e-mail.');
                } else {
                    if ($this->usuarioModel->criar($dados)) {
                        registrarLog('CREATE', "Usuário criado: {$dados['nome']}");
                        setMensagemSucesso('Usuário cadastrado com sucesso!');
                        redirecionar('index.php?c=Usuario&a=index');
                        return;
                    } else {
                        setMensagemErro('Erro ao cadastrar usuário.');
                    }
                }
            }
        }

        include __DIR__ . '/../views/layouts/header.php';
        include __DIR__ . '/../views/layouts/sidebar.php';
        include __DIR__ . '/../views/usuarios/form.php';
        include __DIR__ . '/../views/layouts/footer.php';
    }

    public function editar()
    {
        $id = $_GET['id'] ?? 0;
        $usuario = $this->usuarioModel->buscarPorId($id);
        $perfis = $this->usuarioModel->listarPerfis();

        if (!$usuario) {
            setMensagemErro('Usuário não encontrado.');
            redirecionar('index.php?c=Usuario&a=index');
            return;
        }

        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $dados = [
                'nome' => sanitizar($_POST['nome'] ?? ''),
                'email' => sanitizar($_POST['email'] ?? ''),
                'senha' => $_POST['senha'] ?? '',
                'id_perfil' => intval($_POST['id_perfil'] ?? 0),
                'ativo' => isset($_POST['ativo']) ? 1 : 0
            ];

            if (empty($dados['nome']) || empty($dados['email'])) {
                setMensagemErro('Nome e e-mail são obrigatórios.');
            } elseif (empty($dados['id_perfil'])) {
                setMensagemErro('Selecione um perfil.');
            } else {
                // Verificar se email já existe em outro usuário
                $usuarioExistente = $this->usuarioModel->buscarPorEmail($dados['email']);
                if ($usuarioExistente && $usuarioExistente['id'] != $id) {
                    setMensagemErro('Já existe outro usuário cadastrado com este e-mail.');
                } else {
                    // Se senha foi informada, validar
                    if (!empty($dados['senha'])) {
                        if (strlen($dados['senha']) < PASSWORD_MIN_LENGTH) {
                            setMensagemErro('A senha deve ter no mínimo ' . PASSWORD_MIN_LENGTH . ' caracteres.');
                        } else {
                            // Senha será atualizada
                        }
                    } else {
                        // Não atualizar senha
                        unset($dados['senha']);
                    }

                    if ($this->usuarioModel->atualizar($id, $dados)) {
                        registrarLog('UPDATE', "Usuário atualizado: {$dados['nome']} (ID: {$id})");
                        setMensagemSucesso('Usuário atualizado com sucesso!');
                        redirecionar('index.php?c=Usuario&a=index');
                        return;
                    } else {
                        setMensagemErro('Erro ao atualizar usuário.');
                    }
                }
            }
        }

        include __DIR__ . '/../views/layouts/header.php';
        include __DIR__ . '/../views/layouts/sidebar.php';
        include __DIR__ . '/../views/usuarios/form.php';
        include __DIR__ . '/../views/layouts/footer.php';
    }

    public function excluir()
    {
        $id = $_GET['id'] ?? 0;
        $usuario = $this->usuarioModel->buscarPorId($id);

        if (!$usuario) {
            setMensagemErro('Usuário não encontrado.');
        } elseif ($id == $_SESSION['usuario_id']) {
            setMensagemErro('Você não pode excluir seu próprio usuário.');
        } else {
            if ($this->usuarioModel->excluir($id)) {
                registrarLog('DELETE', "Usuário excluído: {$usuario['nome']} (ID: {$id})");
                setMensagemSucesso('Usuário excluído com sucesso!');
            } else {
                setMensagemErro('Erro ao excluir usuário.');
            }
        }

        redirecionar('index.php?c=Usuario&a=index');
    }
}

