<?php
/**
 * Funções Helper do Sistema
 */

/**
 * Verifica se o usuário está logado
 */
function verificarLogin()
{
    if (!isset($_SESSION['usuario_id']) || empty($_SESSION['usuario_id'])) {
        redirecionar('index.php?c=Auth&a=login');
        exit;
    }
}

/**
 * Verifica se o usuário tem permissão para acessar determinado recurso
 */
function verificarPermissao($perfilPermitido)
{
    verificarLogin();
    
    $perfisPermitidos = is_array($perfilPermitido) ? $perfilPermitido : [$perfilPermitido];
    
    if (!in_array($_SESSION['perfil_id'], $perfisPermitidos)) {
        $_SESSION['erro'] = 'Você não tem permissão para acessar esta página.';
        redirecionar('index.php?c=Dashboard&a=index');
        exit;
    }
}

/**
 * Redireciona para uma URL
 */
function redirecionar($url)
{
    header("Location: " . BASE_URL . $url);
    exit;
}

/**
 * Retorna mensagem de sucesso e limpa da sessão
 */
function getMensagemSucesso()
{
    if (isset($_SESSION['sucesso'])) {
        $mensagem = $_SESSION['sucesso'];
        unset($_SESSION['sucesso']);
        return $mensagem;
    }
    return null;
}

/**
 * Retorna mensagem de erro e limpa da sessão
 */
function getMensagemErro()
{
    if (isset($_SESSION['erro'])) {
        $mensagem = $_SESSION['erro'];
        unset($_SESSION['erro']);
        return $mensagem;
    }
    return null;
}

/**
 * Define mensagem de sucesso
 */
function setMensagemSucesso($mensagem)
{
    $_SESSION['sucesso'] = $mensagem;
}

/**
 * Define mensagem de erro
 */
function setMensagemErro($mensagem)
{
    $_SESSION['erro'] = $mensagem;
}

/**
 * Formata data para exibição
 */
function formatarData($data, $formato = 'd/m/Y')
{
    if (empty($data) || $data === '0000-00-00' || $data === '0000-00-00 00:00:00') {
        return '-';
    }
    
    $timestamp = strtotime($data);
    return $timestamp ? date($formato, $timestamp) : '-';
}

/**
 * Formata valor monetário
 */
function formatarMoeda($valor)
{
    return 'R$ ' . number_format($valor, 2, ',', '.');
}

/**
 * Formata CPF
 */
function formatarCPF($cpf)
{
    $cpf = preg_replace('/[^0-9]/', '', $cpf);
    if (strlen($cpf) === 11) {
        return substr($cpf, 0, 3) . '.' . substr($cpf, 3, 3) . '.' . substr($cpf, 6, 3) . '-' . substr($cpf, 9, 2);
    }
    return $cpf;
}

/**
 * Formata telefone
 */
function formatarTelefone($telefone)
{
    $telefone = preg_replace('/[^0-9]/', '', $telefone);
    if (strlen($telefone) === 11) {
        return '(' . substr($telefone, 0, 2) . ') ' . substr($telefone, 2, 5) . '-' . substr($telefone, 7, 4);
    } elseif (strlen($telefone) === 10) {
        return '(' . substr($telefone, 0, 2) . ') ' . substr($telefone, 2, 4) . '-' . substr($telefone, 6, 4);
    }
    return $telefone;
}

/**
 * Valida CPF
 */
function validarCPF($cpf)
{
    $cpf = preg_replace('/[^0-9]/', '', $cpf);
    
    if (strlen($cpf) != 11) {
        return false;
    }
    
    if (preg_match('/(\d)\1{10}/', $cpf)) {
        return false;
    }
    
    for ($t = 9; $t < 11; $t++) {
        for ($d = 0, $c = 0; $c < $t; $c++) {
            $d += $cpf[$c] * (($t + 1) - $c);
        }
        $d = ((10 * $d) % 11) % 10;
        if ($cpf[$c] != $d) {
            return false;
        }
    }
    
    return true;
}

/**
 * Sanitiza entrada de dados
 */
function sanitizar($dados)
{
    if (is_array($dados)) {
        return array_map('sanitizar', $dados);
    }
    return htmlspecialchars(strip_tags(trim($dados)), ENT_QUOTES, 'UTF-8');
}

/**
 * Registra ação no log de auditoria
 */
function registrarLog($acao, $descricao, $idUsuario = null)
{
    $idUsuario = $idUsuario ?? $_SESSION['usuario_id'] ?? null;
    $ip = $_SERVER['REMOTE_ADDR'] ?? null;
    
    $db = Database::getConnection();
    $sql = "INSERT INTO logs_auditoria (id_usuario, acao, descricao, ip) 
            VALUES (:id_usuario, :acao, :descricao, :ip)";
    
    $stmt = $db->prepare($sql);
    $stmt->bindValue(':id_usuario', $idUsuario);
    $stmt->bindValue(':acao', $acao);
    $stmt->bindValue(':descricao', $descricao);
    $stmt->bindValue(':ip', $ip);
    
    return $stmt->execute();
}

/**
 * Retorna o nome do perfil pelo ID
 */
function getNomePerfil($idPerfil)
{
    $perfis = [
        1 => 'Administrador Geral',
        2 => 'Administrador de Campo/Distrito',
        3 => 'Secretaria',
        4 => 'Tesouraria',
        5 => 'Pastor',
        6 => 'Visualizador/Leitor'
    ];
    
    return $perfis[$idPerfil] ?? 'Desconhecido';
}

