<?php
require_once __DIR__ . '/../config/database.php';

class MovimentoFinanceiro
{
    private $db;

    public function __construct()
    {
        $this->db = Database::getConnection();
    }

    public function listarTodos($filtros = [])
    {
        $sql = "SELECT mf.*, 
                cc.nome AS centro_custo_nome, cc.tipo AS centro_custo_tipo,
                i.nome AS igreja_nome,
                u.nome AS criado_por_nome
                FROM movimentos_financeiros mf
                INNER JOIN centros_custo cc ON cc.id = mf.id_centro_custo
                LEFT JOIN igrejas i ON i.id = mf.id_igreja
                INNER JOIN usuarios u ON u.id = mf.criado_por
                WHERE 1=1";
        
        $params = [];
        
        if (!empty($filtros['data_inicio'])) {
            $sql .= " AND mf.data_movimento >= :data_inicio";
            $params[':data_inicio'] = $filtros['data_inicio'];
        }
        
        if (!empty($filtros['data_fim'])) {
            $sql .= " AND mf.data_movimento <= :data_fim";
            $params[':data_fim'] = $filtros['data_fim'];
        }
        
        if (!empty($filtros['tipo'])) {
            $sql .= " AND mf.tipo = :tipo";
            $params[':tipo'] = $filtros['tipo'];
        }
        
        if (!empty($filtros['id_igreja'])) {
            $sql .= " AND mf.id_igreja = :id_igreja";
            $params[':id_igreja'] = $filtros['id_igreja'];
        }
        
        if (!empty($filtros['id_centro_custo'])) {
            $sql .= " AND mf.id_centro_custo = :id_centro_custo";
            $params[':id_centro_custo'] = $filtros['id_centro_custo'];
        }
        
        $sql .= " ORDER BY mf.data_movimento DESC, mf.criado_em DESC";
        
        $stmt = $this->db->prepare($sql);
        foreach ($params as $key => $value) {
            $stmt->bindValue($key, $value);
        }
        $stmt->execute();
        
        return $stmt->fetchAll();
    }

    public function buscarPorId($id)
    {
        $sql = "SELECT mf.*, 
                cc.nome AS centro_custo_nome, cc.tipo AS centro_custo_tipo,
                i.nome AS igreja_nome,
                u.nome AS criado_por_nome
                FROM movimentos_financeiros mf
                INNER JOIN centros_custo cc ON cc.id = mf.id_centro_custo
                LEFT JOIN igrejas i ON i.id = mf.id_igreja
                INNER JOIN usuarios u ON u.id = mf.criado_por
                WHERE mf.id = :id";
        
        $stmt = $this->db->prepare($sql);
        $stmt->bindValue(':id', $id);
        $stmt->execute();
        
        return $stmt->fetch();
    }

    public function criar($dados)
    {
        $sql = "INSERT INTO movimentos_financeiros 
                (data_movimento, tipo, valor, descricao, id_centro_custo, id_igreja, forma_pagamento, observacoes, criado_por) 
                VALUES (:data_movimento, :tipo, :valor, :descricao, :id_centro_custo, :id_igreja, :forma_pagamento, :observacoes, :criado_por)";
        
        $stmt = $this->db->prepare($sql);
        $stmt->bindValue(':data_movimento', $dados['data_movimento']);
        $stmt->bindValue(':tipo', $dados['tipo']);
        $stmt->bindValue(':valor', $dados['valor']);
        $stmt->bindValue(':descricao', $dados['descricao'] ?? null);
        $stmt->bindValue(':id_centro_custo', $dados['id_centro_custo']);
        $stmt->bindValue(':id_igreja', !empty($dados['id_igreja']) ? $dados['id_igreja'] : null);
        $stmt->bindValue(':forma_pagamento', $dados['forma_pagamento'] ?? null);
        $stmt->bindValue(':observacoes', $dados['observacoes'] ?? null);
        $stmt->bindValue(':criado_por', $dados['criado_por']);
        
        return $stmt->execute();
    }

    public function atualizar($id, $dados)
    {
        $sql = "UPDATE movimentos_financeiros SET 
                data_movimento = :data_movimento, 
                tipo = :tipo, 
                valor = :valor, 
                descricao = :descricao, 
                id_centro_custo = :id_centro_custo, 
                id_igreja = :id_igreja, 
                forma_pagamento = :forma_pagamento, 
                observacoes = :observacoes,
                atualizado_em = NOW()
                WHERE id = :id";
        
        $stmt = $this->db->prepare($sql);
        $stmt->bindValue(':data_movimento', $dados['data_movimento']);
        $stmt->bindValue(':tipo', $dados['tipo']);
        $stmt->bindValue(':valor', $dados['valor']);
        $stmt->bindValue(':descricao', $dados['descricao'] ?? null);
        $stmt->bindValue(':id_centro_custo', $dados['id_centro_custo']);
        $stmt->bindValue(':id_igreja', !empty($dados['id_igreja']) ? $dados['id_igreja'] : null);
        $stmt->bindValue(':forma_pagamento', $dados['forma_pagamento'] ?? null);
        $stmt->bindValue(':observacoes', $dados['observacoes'] ?? null);
        $stmt->bindValue(':id', $id);
        
        return $stmt->execute();
    }

    public function excluir($id)
    {
        $sql = "DELETE FROM movimentos_financeiros WHERE id = :id";
        $stmt = $this->db->prepare($sql);
        $stmt->bindValue(':id', $id);
        return $stmt->execute();
    }

    public function calcularSaldo($filtros = [])
    {
        $sql = "SELECT 
                SUM(CASE WHEN tipo = 'ENTRADA' THEN valor ELSE 0 END) AS total_entradas,
                SUM(CASE WHEN tipo = 'SAIDA' THEN valor ELSE 0 END) AS total_saidas,
                SUM(CASE WHEN tipo = 'ENTRADA' THEN valor ELSE -valor END) AS saldo
                FROM movimentos_financeiros
                WHERE 1=1";
        
        $params = [];
        
        if (!empty($filtros['data_inicio'])) {
            $sql .= " AND data_movimento >= :data_inicio";
            $params[':data_inicio'] = $filtros['data_inicio'];
        }
        
        if (!empty($filtros['data_fim'])) {
            $sql .= " AND data_movimento <= :data_fim";
            $params[':data_fim'] = $filtros['data_fim'];
        }
        
        if (!empty($filtros['id_igreja'])) {
            $sql .= " AND id_igreja = :id_igreja";
            $params[':id_igreja'] = $filtros['id_igreja'];
        }
        
        $stmt = $this->db->prepare($sql);
        foreach ($params as $key => $value) {
            $stmt->bindValue($key, $value);
        }
        $stmt->execute();
        
        $result = $stmt->fetch();
        return [
            'entradas' => floatval($result['total_entradas'] ?? 0),
            'saidas' => floatval($result['total_saidas'] ?? 0),
            'saldo' => floatval($result['saldo'] ?? 0)
        ];
    }

    public function resumoPorCentroCusto($filtros = [])
    {
        $sql = "SELECT 
                cc.nome AS centro_custo,
                cc.tipo,
                SUM(CASE WHEN mf.tipo = 'ENTRADA' THEN mf.valor ELSE 0 END) AS total_entradas,
                SUM(CASE WHEN mf.tipo = 'SAIDA' THEN mf.valor ELSE 0 END) AS total_saidas
                FROM centros_custo cc
                LEFT JOIN movimentos_financeiros mf ON mf.id_centro_custo = cc.id";
        
        $where = [];
        $params = [];
        
        if (!empty($filtros['data_inicio'])) {
            $where[] = "mf.data_movimento >= :data_inicio";
            $params[':data_inicio'] = $filtros['data_inicio'];
        }
        
        if (!empty($filtros['data_fim'])) {
            $where[] = "mf.data_movimento <= :data_fim";
            $params[':data_fim'] = $filtros['data_fim'];
        }
        
        if (!empty($filtros['id_igreja'])) {
            $where[] = "mf.id_igreja = :id_igreja";
            $params[':id_igreja'] = $filtros['id_igreja'];
        }
        
        if (!empty($where)) {
            $sql .= " WHERE " . implode(" AND ", $where);
        }
        
        $sql .= " GROUP BY cc.id, cc.nome, cc.tipo ORDER BY cc.nome";
        
        $stmt = $this->db->prepare($sql);
        foreach ($params as $key => $value) {
            $stmt->bindValue($key, $value);
        }
        $stmt->execute();
        
        return $stmt->fetchAll();
    }
}

