-- ============================================
-- Sistema de Gestão de Igrejas (SGI)
-- Script de Criação do Banco de Dados
-- ============================================

CREATE DATABASE IF NOT EXISTS sgi_db CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
USE sgi_db;

-- ============================================
-- Tabela de Perfis de Usuário
-- ============================================
CREATE TABLE perfis (
  id INT AUTO_INCREMENT PRIMARY KEY,
  nome VARCHAR(100) NOT NULL,
  descricao VARCHAR(255) NULL,
  criado_em DATETIME DEFAULT CURRENT_TIMESTAMP
);

INSERT INTO perfis (id, nome, descricao) VALUES
(1, 'ADMIN_GERAL', 'Administrador Geral - Acesso total ao sistema'),
(2, 'ADMIN_CAMPO', 'Administrador de Campo/Distrito'),
(3, 'SECRETARIA', 'Secretaria - Gestão de cadastros e transferências'),
(4, 'TESOURARIA', 'Tesouraria - Gestão financeira'),
(5, 'PASTOR', 'Pastor - Visualização de informações'),
(6, 'VISUALIZADOR', 'Visualizador/Leitor - Apenas leitura');

-- ============================================
-- Tabela de Usuários
-- ============================================
CREATE TABLE usuarios (
  id INT AUTO_INCREMENT PRIMARY KEY,
  nome VARCHAR(150) NOT NULL,
  email VARCHAR(150) NOT NULL UNIQUE,
  senha VARCHAR(255) NOT NULL,
  id_perfil INT NOT NULL,
  ativo TINYINT(1) DEFAULT 1,
  ultimo_acesso DATETIME NULL,
  criado_em DATETIME DEFAULT CURRENT_TIMESTAMP,
  atualizado_em DATETIME NULL,
  FOREIGN KEY (id_perfil) REFERENCES perfis(id)
);

-- Usuário padrão: admin@admin.com / senha: admin123
INSERT INTO usuarios (nome, email, senha, id_perfil) VALUES
('Administrador', 'admin@admin.com', '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', 1);

-- ============================================
-- Tabela de Igrejas
-- ============================================
CREATE TABLE igrejas (
  id INT AUTO_INCREMENT PRIMARY KEY,
  nome VARCHAR(150) NOT NULL,
  cnpj VARCHAR(20) NULL,
  endereco VARCHAR(255) NULL,
  cidade VARCHAR(100) NULL,
  estado CHAR(2) NULL,
  cep VARCHAR(10) NULL,
  telefone VARCHAR(20) NULL,
  email VARCHAR(150) NULL,
  ativo TINYINT(1) DEFAULT 1,
  criado_em DATETIME DEFAULT CURRENT_TIMESTAMP,
  atualizado_em DATETIME NULL
);

-- ============================================
-- Tabela de Pastores
-- ============================================
CREATE TABLE pastores (
  id INT AUTO_INCREMENT PRIMARY KEY,
  nome VARCHAR(150) NOT NULL,
  cpf VARCHAR(14) UNIQUE NOT NULL,
  rg VARCHAR(20) NULL,
  data_nascimento DATE NULL,
  estado_civil VARCHAR(50) NULL,
  telefone VARCHAR(20) NULL,
  email VARCHAR(150) NULL,
  endereco VARCHAR(255) NULL,
  cidade VARCHAR(100) NULL,
  estado CHAR(2) NULL,
  cep VARCHAR(10) NULL,
  id_igreja_atual INT NULL,
  data_ordenacao DATE NULL,
  observacoes TEXT NULL,
  ativo TINYINT(1) DEFAULT 1,
  criado_em DATETIME DEFAULT CURRENT_TIMESTAMP,
  atualizado_em DATETIME NULL,
  FOREIGN KEY (id_igreja_atual) REFERENCES igrejas(id) ON DELETE SET NULL
);

-- ============================================
-- Tabela de Obreiros
-- ============================================
CREATE TABLE obreiros (
  id INT AUTO_INCREMENT PRIMARY KEY,
  nome VARCHAR(150) NOT NULL,
  cpf VARCHAR(14) UNIQUE NULL,
  tipo VARCHAR(50) NOT NULL,
  telefone VARCHAR(20) NULL,
  email VARCHAR(150) NULL,
  endereco VARCHAR(255) NULL,
  cidade VARCHAR(100) NULL,
  estado CHAR(2) NULL,
  cep VARCHAR(10) NULL,
  id_igreja_atual INT NULL,
  data_consagracao DATE NULL,
  observacoes TEXT NULL,
  ativo TINYINT(1) DEFAULT 1,
  criado_em DATETIME DEFAULT CURRENT_TIMESTAMP,
  atualizado_em DATETIME NULL,
  FOREIGN KEY (id_igreja_atual) REFERENCES igrejas(id) ON DELETE SET NULL
);

-- ============================================
-- Tabela de Transferências de Pastores
-- ============================================
CREATE TABLE transferencias_pastores (
  id INT AUTO_INCREMENT PRIMARY KEY,
  id_pastor INT NOT NULL,
  id_igreja_origem INT NULL,
  id_igreja_destino INT NOT NULL,
  data_transferencia DATE NOT NULL,
  motivo TEXT NULL,
  observacoes TEXT NULL,
  documento_anexo VARCHAR(255) NULL,
  criado_por INT NOT NULL,
  criado_em DATETIME DEFAULT CURRENT_TIMESTAMP,
  FOREIGN KEY (id_pastor) REFERENCES pastores(id) ON DELETE CASCADE,
  FOREIGN KEY (id_igreja_origem) REFERENCES igrejas(id) ON DELETE SET NULL,
  FOREIGN KEY (id_igreja_destino) REFERENCES igrejas(id) ON DELETE RESTRICT,
  FOREIGN KEY (criado_por) REFERENCES usuarios(id)
);

-- ============================================
-- Tabela de Transferências de Obreiros
-- ============================================
CREATE TABLE transferencias_obreiros (
  id INT AUTO_INCREMENT PRIMARY KEY,
  id_obreiro INT NOT NULL,
  id_igreja_origem INT NULL,
  id_igreja_destino INT NOT NULL,
  data_transferencia DATE NOT NULL,
  motivo TEXT NULL,
  observacoes TEXT NULL,
  documento_anexo VARCHAR(255) NULL,
  criado_por INT NOT NULL,
  criado_em DATETIME DEFAULT CURRENT_TIMESTAMP,
  FOREIGN KEY (id_obreiro) REFERENCES obreiros(id) ON DELETE CASCADE,
  FOREIGN KEY (id_igreja_origem) REFERENCES igrejas(id) ON DELETE SET NULL,
  FOREIGN KEY (id_igreja_destino) REFERENCES igrejas(id) ON DELETE RESTRICT,
  FOREIGN KEY (criado_por) REFERENCES usuarios(id)
);

-- ============================================
-- Tabela de Centros de Custo
-- ============================================
CREATE TABLE centros_custo (
  id INT AUTO_INCREMENT PRIMARY KEY,
  nome VARCHAR(100) NOT NULL,
  tipo ENUM('ENTRADA', 'SAIDA') NOT NULL,
  descricao VARCHAR(255) NULL,
  ativo TINYINT(1) DEFAULT 1,
  criado_em DATETIME DEFAULT CURRENT_TIMESTAMP
);

-- Inserir centros de custo padrão
INSERT INTO centros_custo (nome, tipo, descricao) VALUES
('Dízimos', 'ENTRADA', 'Recebimento de dízimos'),
('Ofertas', 'ENTRADA', 'Recebimento de ofertas'),
('Missões', 'ENTRADA', 'Contribuições para missões'),
('Eventos', 'ENTRADA', 'Receitas de eventos'),
('Outras Receitas', 'ENTRADA', 'Outras receitas diversas'),
('Salários', 'SAIDA', 'Pagamento de salários'),
('Manutenção', 'SAIDA', 'Gastos com manutenção'),
('Construção', 'SAIDA', 'Gastos com construção/reforma'),
('Eventos', 'SAIDA', 'Gastos com eventos'),
('Missões', 'SAIDA', 'Contribuições para missões'),
('Outras Despesas', 'SAIDA', 'Outras despesas diversas');

-- ============================================
-- Tabela de Movimentos Financeiros
-- ============================================
CREATE TABLE movimentos_financeiros (
  id INT AUTO_INCREMENT PRIMARY KEY,
  data_movimento DATE NOT NULL,
  tipo ENUM('ENTRADA', 'SAIDA') NOT NULL,
  valor DECIMAL(10, 2) NOT NULL,
  descricao VARCHAR(255) NULL,
  id_centro_custo INT NOT NULL,
  id_igreja INT NULL,
  forma_pagamento VARCHAR(50) NULL,
  observacoes TEXT NULL,
  criado_por INT NOT NULL,
  criado_em DATETIME DEFAULT CURRENT_TIMESTAMP,
  atualizado_em DATETIME NULL,
  FOREIGN KEY (id_centro_custo) REFERENCES centros_custo(id),
  FOREIGN KEY (id_igreja) REFERENCES igrejas(id) ON DELETE SET NULL,
  FOREIGN KEY (criado_por) REFERENCES usuarios(id)
);

-- ============================================
-- Tabela de Logs de Auditoria
-- ============================================
CREATE TABLE logs_auditoria (
  id INT AUTO_INCREMENT PRIMARY KEY,
  id_usuario INT NOT NULL,
  acao VARCHAR(100) NOT NULL,
  descricao TEXT NOT NULL,
  ip VARCHAR(45) NULL,
  user_agent VARCHAR(255) NULL,
  criado_em DATETIME DEFAULT CURRENT_TIMESTAMP,
  FOREIGN KEY (id_usuario) REFERENCES usuarios(id)
);

-- ============================================
-- Índices para melhor performance
-- ============================================
CREATE INDEX idx_pastores_igreja ON pastores(id_igreja_atual);
CREATE INDEX idx_obreiros_igreja ON obreiros(id_igreja_atual);
CREATE INDEX idx_transferencias_pastor ON transferencias_pastores(id_pastor);
CREATE INDEX idx_transferencias_obreiro ON transferencias_obreiros(id_obreiro);
CREATE INDEX idx_movimentos_data ON movimentos_financeiros(data_movimento);
CREATE INDEX idx_movimentos_igreja ON movimentos_financeiros(id_igreja);
CREATE INDEX idx_logs_usuario ON logs_auditoria(id_usuario);
CREATE INDEX idx_logs_acao ON logs_auditoria(acao);

